/**
* \file: InputSourceReport.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: M. Adachi / ADITJ/SW / madachi@jp.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <adit_logging.h>
#include <errno.h>
#include "InputSourceReport.h"

LOG_IMPORT_CONTEXT(aauto_input)

namespace adit { namespace aauto
{

InputReport::InputReport()
{
    if (0 != pthread_mutex_init(&mInputReportMutex , NULL)) {
        assert(&mInputReportMutex);
    }

    inputSource = nullptr;
    mRunning.store(false, std::memory_order_release);

    width = 0;
    height = 0;
    marginX = 0;
    marginY = 0;
}

InputReport::~InputReport()
{
    /* Attempting to unlock a mutex that it has not locked
     * or a mutex which is unlocked, results in undefined behavior */
    int  err = pthread_mutex_lock(&mInputReportMutex);
    if (0 == err)
    {
        // Attempting to destroy a locked mutex results in undefined behavior
        err = pthread_mutex_unlock(&mInputReportMutex);
        if (err != 0)
        {
            LOG_ERROR((aauto_input, "InputReport::~InputReport() Failed to unlock mutex (err=%d, errno=%d)", err, errno));
        }
    }
    else
    {
        LOG_ERROR((aauto_input, "InputReport::~InputReport() Failed to lock mutex (err=%d, errno=%d)", err, errno));
    }
    /* destroy the mutex object referenced by mMutex */
    pthread_mutex_destroy(&mInputReportMutex);
}

void InputReport::initialize(InputSource* inInputSource, int touchWidth, int touchHeight, int inMarginX, int inMarginY)
{
    inputSource = inInputSource;
    width = touchWidth;
    height = touchHeight;
    marginX = inMarginX;
    marginY = inMarginY;
    mRunning.store(true, std::memory_order_release);
}

bool InputReport::shutdown()
{
    bool ret = true;

    int err = pthread_mutex_lock(&mInputReportMutex);
    if (0 == err)
    {
        if (true == mRunning.load(std::memory_order_acquire))
        {
            mRunning.store(false, std::memory_order_release);
        }
        else
        {
            LOG_INFO((aauto_input, "InputReport::shutdown() Not running. Was already stopped"));
        }

        // unlock mutex only when we got the lock
        err = pthread_mutex_unlock(&mInputReportMutex);
        if (0 != err)
        {
            LOG_ERROR((aauto_input, "InputReport::shutdown() Failed to unlock mutex (err=%d, errno=%d)", err, errno));
        }
    }
    else
    {
        LOG_ERROR((aauto_input, "InputReport::shutdown() Failed to lock mutex (err=%d, errno=%d)", err, errno));
    }

    return ret;
}

void InputReport::setTouch(uint32_t id, double x, double y, TouchActionType action)
{
     touch.pointerIds    = id;
     touch.x             = x;
     touch.y             = y;
     touch.action        = action;
}

bool InputReport::SendInput()
{
    uint32_t id = touch.pointerIds;
    bool ret = false;
    int err = 0;

    LOGD_VERBOSE((aauto_input, "InputReport::SendInput() entry: action: %d, finger: %d, x: %f, y: %f",
                  touch.action, touch.pointerIds, touch.x, touch.y));

   err = pthread_mutex_lock(&mInputReportMutex);
   if (0 == err)
   {
        auto it = mTrackedPointers.find(id);
        // finger touching display, therefore tracked
        if (it != mTrackedPointers.end())
        {
            switch (touch.action)
            {
            case TActionDown:
                LOG_WARN((aauto_input, "TouchDown event: Didn't catch any up event"));
                it->second.x = (touch.x * width) - marginX;
                it->second.y = (touch.y * height) - marginY;
                if (mTrackedPointers.size() == 1)
                {
                    // first finger down
                    ret = SendInputReport(ACTION_DOWN, id);
                }
                else
                {
                    // 2nd-nth finger down
                    ret = SendInputReport(ACTION_POINTER_DOWN, id);
                }
                break;
            case TActionMotion:
                LOGD_VERBOSE((aauto_input, "TouchMotion event"));
                it->second.x = (touch.x * width) - marginX;
                it->second.y = (touch.y * height) - marginY;
                ret = SendInputReport(ACTION_MOVED, id);
                break;
            case TActionUp:
                LOGD_VERBOSE((aauto_input, "TouchUp event"));
                if (mTrackedPointers.size() == 1)
                {
                    // first finger up
                    ret = SendInputReport(ACTION_UP, id);
                }
                else
                {
                    // 2nd-nth finger up
                    ret = SendInputReport(ACTION_POINTER_UP, id);
                }
                mTrackedPointers.erase(id);
                break;
            }
        }
        else
        {
            // new finger
            switch (touch.action)
            {
            case TActionDown:
                LOGD_VERBOSE((aauto_input, "TouchDown event: New finger"));
                mTrackedPointers.insert(std::pair<int, Pointer>(id, {(uint32_t)((touch.x * width) - marginX), (uint32_t)((touch.y * height) - marginY)}));
                if (mTrackedPointers.size() == 1)
                {
                    // first finger down
                    ret = SendInputReport(ACTION_DOWN, id);
                }
                else
                {
                    // 2nd-nth finger up
                    ret = SendInputReport(ACTION_POINTER_DOWN, id);
                }
                break;
            case TActionMotion:
                LOG_WARN((aauto_input, "TouchMotion event: Finger can't move when it wasn't contacting the screen"));
                break;
            case TActionUp:
                LOG_WARN((aauto_input, "TouchUp event: Finger can't be lifted when it wasn't contacting the screen"));
                break;
            }
        }

        err = pthread_mutex_unlock(&mInputReportMutex);
        if (0  != err)
        {
            LOG_ERROR((aauto_input, "InputReport::SendInput() Failed to unlock mutex (err=%d, errno=%d)", err, errno));
        }
    }
    else
    {
        LOG_ERROR((aauto_input, "InputReport::SendInput() Failed to lock mutex (err=%d, errno=%d)", err, errno));
    }

    return ret;
}

bool InputReport::SendInputReport(int action, int id)
{
    bool ret = 0;
    struct timespec timestamp;
    int actionIndex = 0;
    int eventIndex = 0;

    // Access to mTrackedPointers is protected because SendInputReport() called while mutex is locked.
    uint32_t x[mTrackedPointers.size()];
    uint32_t y[mTrackedPointers.size()];
    uint32_t ids[mTrackedPointers.size()];

    clock_gettime(CLOCK_MONOTONIC, &timestamp);

    for (auto it = mTrackedPointers.begin(); it != mTrackedPointers.end(); it++, eventIndex++)
    {
        x[eventIndex] = it->second.x;
        y[eventIndex] = it->second.y;
        ids[eventIndex] = it->first;
        if (id == it->first)
        {
            actionIndex = eventIndex;
        }

        LOGD_VERBOSE((aauto_input, "Report Touch[%d] x = %u y = %u id = %u (count = %d)"
                , eventIndex, x[eventIndex], y[eventIndex], ids[eventIndex], (int)mTrackedPointers.size()));
    }

    if (true == mRunning.load(std::memory_order_acquire)) {
        ret = inputSource->reportTouch((timestamp.tv_sec * 1000000000) + timestamp.tv_nsec, mTrackedPointers.size(), ids, x, y, action, actionIndex);
        return ret;
    } else {
        LOG_WARN((aauto_input, "Touch not reported due to shutdown"));
        return false;
    }
}

} } /* namespace adit { namespace aauto */
